#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Créé fév 2026   @auteur: David ALBERTO (www.astrolabe-science.fr)
Trace une courbe théorique de titrage conductimétrique.
En option, exporte les données calculées dans un fichier csv.
"""
import matplotlib.pyplot as plt
import numpy as np
import pandas as pd  # export csv

plt.rcParams["font.family"] = "Roboto"
couleurligne = "blue"

# =============================================================================
# Données initiales
# =============================================================================
"""
A1 : ion titré
K1 : spectateur accompagnant A1
K2 : ion titrant
A2 : spectateur accompagnant K2
"""
C1 = 0.0026# concentration solution titrée (mol/L)
C2 = 0.0080# concentration solution titrante (mol/L)
V1 = 20 # volume initial solution titrée (mL)
Veau = 250  # mL ajout d'eau initial
#   Conductivités molaires ioniques (mS.m2/mol):
lambdaA1 = 7.63  # ion titré
lambdaK1 = 80.00  # ion spectateur
lambdaK2 = 6.2  # ion titrant
lambdaA2 = 7.14  # ion spectateur
#
#  nombres stoechiométriques :
nu1 = 1
nu2 = 1
#
# incrément de volume (mL) :
DV2 = 0.5
V2final = 12  # mL de volume titrant versé
# =============================================================================
# CALCULS
# =============================================================================
C1 = C1 * 1e3  # conversion mol/m3
C2 = C2 * 1e3  # conversion mol/m3
V1 = V1 /1e6  # conversion m3
DV2 = DV2 /1e6  # conversion m3
Veau = Veau /1e6  # conversion m3
V2final = V2final /1e6  # conversion m3
Veq = nu2 / nu1 * C1 * V1 / C2
V2avant = np.arange(0, Veq, DV2)  # liste des volumes versés avant l'équiv
Vtotal = V1 + Veau + V2avant
# avancement final
xf = nu1 / nu2 * C2 * V2avant
# concentrations :
CA1 = (C1 * V1 - xf) / Vtotal
CK1 = (C1 * V1) / Vtotal
CA2 = (C2 * V2avant) / Vtotal

conductivite_avant = lambdaA1 * CA1 + lambdaK1 * CK1 + lambdaA2 * CA2

    #  après l'équivalence :
V2apres = np.arange(Veq, V2final, DV2)
Vtotal = V1 + Veau + V2apres

CK1 = (C1 * V1) / Vtotal
CK2 = (V2apres - Veq) * C2 / Vtotal
CA2 = (C2 * V2apres) / Vtotal
conductivite_apres = lambdaK2 * CK2 + lambdaK1 * CK1 + lambdaA2 * CA2

# =============================================================================
# FONCTIONS :
# =============================================================================
def trace(x, y):
    """
    trace y en fonction de x, d'abord une ligne continue, puis des points
    """
    ax.plot(x, y, c=couleurligne)  # ligne1
    ax.scatter(x, y, fc='orange', ec=couleurligne, zorder=2) # points

def exportcsv(x, y, nomFichier):
    df = pd.DataFrame({'V': x, 'sigma': y})
    df.to_csv(nomFichier + '.csv', index=False, sep=' ')
    print(df)

# =============================================================================
# GRAPHIQUE
# =============================================================================

fig = plt.figure(figsize=(6,4))
ax = plt.subplot(111)

ax.set(
        # ylim=(0, ymax)
        xlabel="volume de solution titrante versée (mL)",
        ylabel = 'conductivité (mS/m)',
        )
ax.set_title("Titrage conductimétrique", fontweight='bold')
#  reconversion en mL :
V2avant = V2avant * 1e6
V2apres = V2apres * 1e6
Veq = Veq * 1e6

#  regroupement des données :
V2verse = np.concatenate((V2avant, V2apres))
conductivite = np.concatenate((conductivite_avant,conductivite_apres))

trace(V2verse, conductivite)

ax.set_xticks(np.arange(0, V2final*1e6, 5))
ax.set_xticks(np.arange(0, V2final*1e6, 1), minor=True)
# ax.grid()


ax.text(0.01,0.98, f"Veq = {Veq:.1f} mL".replace('.',','), va='top', ha='left',
        transform = ax.transAxes)

exportcsv(V2verse, conductivite, "titrage_conductimetrique")

fig.savefig('titrage_conductimetrique.png', dpi=250)
plt.show()
