#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Créé Jan 2 2026
@auteur: David ALBERTO (www.astrolabe-science.fr)
Calendrier des phases de la Lune pour une année.
Moon calendar for the year.
"""
import matplotlib.pyplot as plt
import matplotlib.patches as patches
import numpy as np
import pandas as pd
from skyfield.api import load
from skyfield.framelib import ecliptic_frame
from skyfield import almanac

plt.rcParams["font.family"] = "Linux Libertine O"  # ou autre police système

# =============================================================================
# Paramètres à personnaliser
# =============================================================================
annee = 2026
R = 0.5  # rayon disque lunaire
xfact = 2.5 # facteur d'échelle en x
yfact = 1.4 # facteur d'échelle en y
figwidth = 21  # cm
figheight = 29.7  # cm
liste_mois = ['Jan', 'Fév', 'Mar', 'Avr', 'Mai', 'Juin', 'Juil',
              'Aou', 'Sept', 'Oct', 'Nov', 'Déc']

prop_PL = dict(radius = R*1.3, fc='gainsboro', lw=0.0)  # format PL
prop_NL = dict(radius = R*1.3, fc='orange', lw=0.0, alpha = 0.5)  # format NL
ylegend = -2.5  # position verticale de la légende
# =============================================================================
# # FONCTIONS
# =============================================================================

def phase(t):
    """
    t : date Skyfield
    renvoie phase en degrés, et pourcentage d'éclairement du disque lunaire
    """
    e = earth.at(t)
    s = e.observe(sun).apparent()
    m = e.observe(moon).apparent()
    _, slon, _ = s.frame_latlon(ecliptic_frame)
    _, mlon, _ = m.frame_latlon(ecliptic_frame)
    phase = (mlon.degrees - slon.degrees) % 360.0
    percent = 100.0 * m.fraction_illuminated(sun)
    return phase, percent

def disque(phi, pourcent, j, m):
    """
    phi : phase (degrés)
    pourcent : float. pourcentage d'éclairement
    j, m : quantièmes du jour et du mois.
    Trace le disque lunaire.
    """
    disqueblanc = patches.Circle((m*xfact, j*yfact), radius=R, fc='white',
                                 ec='k', lw=0.3)
    ax.add_patch(disqueblanc)
    # demi-disque sombre :
    if phi < 180:
        darksemidisc = patches.Wedge((m*xfact, j*yfact), R, theta1=90, theta2=270,
                                 fc='k')
    else:
        darksemidisc = patches.Wedge((m*xfact, j*yfact), R, theta1=270, theta2=90,
                                 fc='k')
    ax.add_patch(darksemidisc)
    #  ellipse
    if pourcent >= 50:
        ellipsecolor='white'
    else:
        ellipsecolor='k'
    b = R * np.cos(np.radians(phi))
    ellipse = patches.Ellipse((m*xfact, j*yfact), b*2, (R-0.0003)*2, fc=ellipsecolor, ec=None,
                              lw=0.0, zorder=1)
    ax.add_patch(ellipse)

def nommois():
    """
    affiche les noms des mois.
    """
    for i, nom in enumerate(liste_mois):
        ax.text((i+1)*xfact, -1*yfact, nom, ha='center', va='center')
        ax.text((i+1)*xfact, 31.2*yfact, nom, ha='center', va='center')

def numjours():
    #  affiche les numéros des jours
    for j in range(1, 32):
        ax.text(0.5*xfact, (j-1)*yfact, j, ha='center', va='center')
        ax.text(12.5*xfact, (j-1)*yfact, j, ha='center', va='center')

def titre():  # affiche le titre
    ax0.text(0.5, 0.95, f"Calendrier lunaire\n{annee}",
             fontsize=24, fontweight='bold', ha='center', va='center',
             transform=ax0.transAxes)

def NLPL():
    """
    recherche des syzygies avec Skyfield almanac, et affiche les 
    disques correspondant, en fond.
    """
    t1 = ts.utc(annee, 1, 1)
    t2 = ts.utc(annee+1, 1, 1)
    #  recherche des PL et NL de l'année :
    t, num = almanac.find_discrete(t1, t2, almanac.moon_phases(eph))
    for i in range(len(t)):
        x = float(t[i].utc_strftime('%m'))
        y = float(t[i].utc_strftime('%d'))-1
        if num[i] == 0:  # si NL
            disque = patches.Circle((x*xfact, y*yfact),zorder=0, **prop_NL)
        elif num[i] == 2:  # si PL
            disque = patches.Circle((x*xfact, y*yfact),zorder=0, **prop_PL)
        else:
            continue
        ax.add_patch(disque)

def phases():
    """
    trace les phases pour l'année.
    """
    for i in range(len(df)):
        m = df.iloc[i].mois
        j = float(df.iloc[i].jour)
        ph = df.iloc[i].phase
        pourcent = df.iloc[i].pourcent
        disque(ph, pourcent, j-1, m)
# =============================================================================
#
# =============================================================================
  # if .bsp already loaded (customize path):
# eph = load('../skyfield-ephemerides/de421.bsp') # range 1900 to 2050
#  for other time ranges, see https://rhodesmill.org/skyfield/planets.html
#  if .bsp file not loaded yet (will download it once):
eph = load('de421.bsp') # range 1900 to 2050
sun, moon, earth = eph['sun'], eph['moon'], eph['earth']
ts = load.timescale()

nbjours = int(ts.utc(annee+1,1,1) - ts.utc(annee,1,1))  # number of days in year

periode = ts.utc(annee, 1, range(1, nbjours+1), 12)
phase, pourcent = phase(periode)

mois = [float(periode[i].utc_strftime('%m')) for i in range(len(periode))]
jours = [periode[i].utc_strftime('%d') for i in range(len(periode))]

cols = ['jours', 'mois', 'phase', 'pourcentage']

df = pd.DataFrame({'jour': jours, 'mois': mois, 'phase': phase, 'pourcent':pourcent})

# =============================================================================
# graphique
# =============================================================================

fig = plt.figure(figsize=(figwidth/2.54, figheight/2.54), tight_layout=True)

ax0 = plt.subplot(111)  # axe principal
ax0.set(
        xticks=[],
        yticks=[],
        )
ax0.axis('off')
#  axe secondaire contenant les phases :
ax = ax0.inset_axes([0.1, 0.05, 0.8, 0.85])
ax.set_aspect('equal')
ax.set_ylim(-4*yfact, 31.5*yfact)
ax.invert_yaxis()
ax.set(
       xlim=(0.5, 13*xfact),
        yticks=[],
       )
ax.axis('off')

# =============================================================================
# tracé pour toute l'année :
# =============================================================================
phases()
nommois()
numjours()
titre()
NLPL()

# =============================================================================
# Légende :
# =============================================================================
listephi = np.arange(0, 360, 45)
listefrac = 50 * (1-np.cos(np.radians(listephi)))
listex = range(3,11)
legendes = ['Nouvelle\nLune', 'Premier\ncroissant', 'Premier\nQuartier',
            'Gibbeuse\ncroissante', 'Pleine\nLune', 'Gibbeuse\ndécroissante',
            'Dernier\nQuartier', 'Dernier\ncroissant']

NL = patches.Circle((3*xfact, ylegend*yfact), **prop_NL)
ax.add_patch(NL)
PL = patches.Circle((7*xfact, ylegend*yfact), **prop_PL)
ax.add_patch(PL)

for i in range(len(listephi)):
    disque(listephi[i], listefrac[i], ylegend, listex[i])
    ax.text(listex[i]*xfact, (ylegend-R*2)*yfact, legendes[i],
            ha='center', va='center', fontsize=7)

# =============================================================================
# Signature :
# =============================================================================
ax.text(1.05, 0.05, "D. Alberto (www.astrolabe-science.fr)", rotation=90,
        ha='left', va='bottom', transform=ax.transAxes,
        c='gray', fontsize=9)

plt.show()

fig.savefig('calendrierLunaire' + str(annee) + '.png', dpi=200)
fig.savefig('calendrierLunaire' + str(annee) + '.pdf')
