#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created Dec 2025

@author: David ALBERTO (www.astrolabe-science.fr)

Draws a map on sundials in Europe, with a count of 
the number of sundials per country.
"""

#  calling libraries
import matplotlib.pyplot as plt  # plots
import matplotlib as mpl#Normalisation des valeurs
import matplotlib.colorbar as cbar  # barres de couleurs
# from mpl_toolkits.axes_grid1 import make_axes_locatable  # positionner les barres de couleurs
import geopandas as gpd  # managing geodata

#  personal settings:
plt.rcParams["font.family"] = "Roboto"  # or other system font
plt.rcParams["font.size"] = 8
# crs = 'EPSG:4326'  # projection for Europe
crs = 'ESRI:102014'  # projection for Europe
cmap = plt.get_cmap('GnBu', 10) # colormap, 10 colors for 0-1000 sundials

countries_prop = dict(ec='k', lw=0.15)
dots_prop = dict(ec='k', lw=0.1, fc='orange', alpha=0.5, s=1)

xmin, xmax =-2.4e6, 2.2e6
ymin, ymax = 0.5e6, 5e6

countries_names = [
    'france',
    'austria',
    'italy',
    'spain',
    'portugal',
    'uk',
    'ireland',
    'iceland',
    'poland',
    'romania',
    'czechia',
    'greece',
    'bulgaria',
    'denmark',
    'norway',
    'finland',
    'bosnia',
    'slovakia',
    'slovenia',
    'switzerland',
    'croatia',
    'germany',
    'hungary',
    'sweden',
    'belgium',
    'netherlands',
    'serbia',
    'montenegro',
    'northmacedonia',
    'albania',
    'belarus',
    'latvia',
    'lithuania',
    'estonia',
    'russia',
    'ukraine',
    'moldova',
    ]

# =============================================================================
# functions :
# =============================================================================

def lowerinitial(string):
    name = string[0].lower() + string[-len(string)+1:]
    return name

# =============================================================================
# importing geographic data :
# =============================================================================
    #  coordinates of sundials :
countries = gpd.read_file('world-administrative-boundaries/world-administrative-boundaries.shp')

countries = countries[['name', 'continent', 'geometry']]

# =============================================================================
# re-projecting geodata in a different CRS:
# =============================================================================
countries = countries.to_crs(crs)
# europe = europe.to_crs(crs)

countries = countries.replace('Czech Republic', 'Czechia')
countries = countries.replace('Russian Federation', 'Russia')
countries = countries.replace('Moldova, Republic of', 'Moldova')
countries = countries.replace('Bosnia & Herzegovina', 'Bosnia')
countries = countries.replace('U.K. of Great Britain and Northern Ireland', 'Uk')
countries = countries.replace('The former Yugoslav Republic of Macedonia',
                              'Northmacedonia')

# drop countries not in the list countries_names:
for idx, country in countries.iterrows():
    name = lowerinitial(country['name'])
    if name not in countries_names:
        countries = countries.drop([idx])


#  importing sundial coordinates into the countries DataFrame

sundialcount = []  # list of sundial counts
sundial_coords = []
for idx, country in countries.iterrows():
    name = lowerinitial(country['name'])
    sdf = gpd.read_file(name + '.geojson')  # sundial data in the country
    # re-projecting geodata in a different CRS:
    sdf = sdf.to_crs(crs)
    sundialcount.append(len(sdf))
    sundial_coords.append(sdf.geometry)

countries['count'] = sundialcount
countries['coords'] = sundial_coords
countries['centroid'] = countries.geometry.centroid
total = countries['count'].sum()

norm = mpl.colors.Normalize(vmin=0,vmax=1000) # normalisation
# =============================================================================
# Creating the figure and ax
# =============================================================================

fig = plt.figure(figsize=(5, 5.2),
                  constrained_layout=True
                 )

ax = plt.subplot(111,
                 facecolor='lightsteelblue',
                 alpha=0.1)
ax.set(
        xticks=[],
        yticks=[],
        xlim=(xmin, xmax),
        ylim=(ymin, ymax),
        title="sundials in Europe (on OpenStreetMap)",
        )


# =============================================================================
# Plotting countries
# =============================================================================

countries.plot(ax=ax, **countries_prop)
countries.plot(ax=ax, **countries_prop, column='count',
                cmap=cmap,
                            )

# =============================================================================
# plot sundials:
# =============================================================================
ynorth = 0
for idx, country in countries.iterrows():
    coords = country['coords']
    ax.scatter(coords.x, coords.y, **dots_prop, zorder=2)
    if coords.y.max() > ynorth and coords.y.max() < ymax:
        ynorth = coords.y.max()
        ind = coords.y[coords.y == ynorth].index
        xnorth = coords.x[ind].values[0]

# =============================================================================
# Pointing at the northermost sundial on this map:
# =============================================================================

ax.annotate('northernmost', xy=(xnorth, ynorth),
            xytext=(xnorth+5e5, ynorth+2e5),
            arrowprops=dict(facecolor=cmap(10), shrink=0.05,
                            width=.05, edgecolor=None,
                            linewidth=0.2,
                            headlength=5,
                            headwidth=2,
                            )
            )

# =============================================================================
# ADD COLORBAR :
# =============================================================================
#  creating an axe for the colorbar:
cax = ax.inset_axes([0, -0.1, 1, 0.05], transform=ax.transAxes)
cb = cbar.ColorbarBase(cax, cmap=cmap, norm=norm, orientation='horizontal')
cb.set_label('sundials per country')
cax.set_xticks(range(0,1100,100))


# =============================================================================
#  Option: writing the number of sundials per country on each centroid:
# =============================================================================
for idx, country in countries.iterrows():
    if country['count'] != 0 and country['name'] != 'Russia':
        ax.text(country['centroid'].x, country['centroid'].y, country['count'],
            fontsize=10, fontweight='bold', c='gray', alpha=0.5,
            zorder=1, ha='center', va='center')

ax.text(0.02, 0.02, f"total: {total}", transform=ax.transAxes,
        )

ax.text(0.99, 0.02, "Data: OpenStreetMap", transform=ax.transAxes,
        rotation=90, ha='right', va='bottom', fontsize=6,
        )
ax.text(0.99, 0.98, "D. Alberto (www.astrolabe-science.fr)", transform=ax.transAxes,
        rotation=90, ha='right', va='top', fontsize=6,
        )

# =============================================================================
# INSET with Canaries
# =============================================================================
ax2 = ax.inset_axes([0.32,0.005, 0.2,0.10])

ax2.set(
        xlim=(-3e6, -2.25e6),
        ylim=(0.1e6, 0.5e6),
        facecolor='lightsteelblue',
        xticks=[],
        yticks=[],
        )
ax2.spines['top'].set_linewidth(0.2)
ax2.spines['bottom'].set_linewidth(0.2)
ax2.spines['left'].set_linewidth(0.2)
ax2.spines['right'].set_linewidth(0.2)
spain_sundials = gpd.read_file('spain.geojson')
spain_sundials = spain_sundials.to_crs(crs)
spain = countries[countries.name== 'Spain']

spain.plot(ax=ax2, **countries_prop, column='count',
               cmap=cmap,
                            )
ax2.scatter(spain_sundials.geometry.x, spain_sundials.geometry.y, **dots_prop)

ax2.text(0.98, 0.98, 'Canaries (Sp.)', fontsize=6, ha='right',va='top',
         transform=ax2.transAxes)


fig.savefig('map_sundials_Europe.png', dpi=300)
fig.savefig('map_sundials_Europe.pdf')
