#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created Aug 2025
Author: David Alberto (www.astrolabe.science.fr)
Tracé d'un cadran solaire conique, avec carte du monde.
"""

import matplotlib.pyplot as plt
import numpy as np
import geopandas as gpd

# =============================================================================
# -------------  réglages----------------------------
lat = 49.5  # latitude degrés
longitude = 0.109  # longitude en degrés

# géométrie du cône 3D/geometry of the 3D cone:
r = 1.5  # petit rayon / small radius
R = 12  # grand rayon / big radius
# hcone = 12.12  # hauteur du cône / height of the cone (not working...)
hcone = (R-r) / np.tan(np.radians(30))  # hauteur du cône, pour tracé de 180°
# ---------
#  to avoid lines crossing the map from one side to the other, adjust this parameter:
trimA = 0.9990  # paramètre pour couper les lignes aux bords
  # coordonnées géographiques du point d'affichage de la légende :
      # coordinates of the label
lonleg, latleg = -35, 45 # Atlantique Nord/North Atlantic

couleur_heures = 'mediumblue'
style_lignes_hor = dict(lw=1, c=couleur_heures)
style_demies = dict(lw=0.6, c=couleur_heures, ls='--')
style_arcs_diurnes = dict(lw=0.6, c='yellowgreen')
style_equinoxes = dict(lw=0.8, c='darkolivegreen')
style_decoupe = dict(lw=0.8, c='crimson')
plt.rcParams["font.family"] = "Roboto Condensed"
plt.rcParams["font.size"] = 8
plt.rcParams["lines.solid_capstyle"] = "round"
plt.rcParams["lines.linewidth"] = 0.5
couleur_carte = 'peru'
NomFichier = "cadran_conique"  # figure file name

# ################################################
#  calculs de paramètres astronomiques :
lat_rad = np.radians(lat)  # latitude radians
eps = 23 + 26/60 + 20/3600 #  obliquité de l'écliptique
epsrad = np.radians(eps)
long_rad = np.radians(longitude)
pi = np.pi
Hmax = np.arccos(-np.tan(lat_rad) * np.tan(epsrad))  # angle horaire max
heuremin = (12-np.degrees(Hmax)//15)  # heure minimale
# heuremin = 7
heuremax = (12+np.degrees(Hmax)//15)  # heure maximale
# heuremax = 17
# ################################################
#  calculs de paramètres géométriques du cône aplati :
# a = np.arctan((R - r)/hcone)  # demi-angle au sommet du cône
a = np.arctan2(R - r, hcone)  # demi-angle au sommet du cône
r2 = R / np.sin(a)  # hauteur oblique du cone entier
r1 = r / np.sin(a)  # rayon intérieur du patron
Acone = 360 * np.sin(a)  # angle du patron ouvert
Aconerad = np.radians(Acone)
# ################################################
# gestion des dimensions du graphique :
inch = 2.54  # conversion des cm en pouces
# largeur_axe = 2*(r2 * np.sin(Aconerad/2)) / inch
# hauteur_axe = (r2 - r1) * np.cos(Aconerad/2) / inch
fig_width = 29.7 / inch
fig_height = 21 / inch
largeur_axe = r2 / inch * 4
hauteur_axe = r2 / 2 / inch * 2
marge_G = 0 / inch
marge_D = fig_width - marge_G - largeur_axe
marge_B = marge_G
marge_H = fig_height - marge_B - hauteur_axe
# ==============================================================
#------------- FONCTIONS-----

def azimut(H, decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie l'azimut du Soleil, en radians.
    """
    A = np.arctan2(np.cos(decl) * np.sin(H), np.sin(lat_rad) *
                   np.cos(decl) * np.cos(H) - np.cos(lat_rad) * np.sin(decl))
    return A

def hauteur(H, decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie la hauteur du Soleil, en radians.
    """
    h = np.arcsin(np.sin(decl) * np.sin(lat_rad) +
                  np.cos(decl) * np.cos(lat_rad) * np.cos(H))
    return h

def projection(H, decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie les coordonnées polaires du point d'ombre. Theta en radians.
    """
    A = azimut(H, decl)
    # suppression des données au bord du tracé (pour éviter les traits de carte en travers) :
    A = np.where(abs(A) > Aconerad*trimA, np.nan, A)
    h = hauteur(H, decl)
    rM = R * (1 / np.sin(a) - np.sin(h) / np.cos(h-a))  # 1
    theta = A * np.sin(a)
    return theta, rM


def notnanindex(array):
    """
    array : array numpy.
    renvoie les index de la première et de la
    denière valeur qui ne soit pas un NaN.
    """
    if len(array[np.isfinite(array)]) > 0:
        first = array[np.isfinite(array)][0]
        last = array[np.isfinite(array)][-1]
        idx1 = np.where(array == first)[0][0]
        idx2 = np.where(array == last)[0][0]
    else:
        idx1 = 0
        idx2 = 0
    return idx1, idx2

def getPolyCoords(row, geom, coord_type):
    """
    Returns the coordinates ('x|y') of edges/vertices of a Polygon/others
    Args:
    - row: the row object from a geodataframe; i.e.   df.loc[1]
    - geom: the name of "geometry" column, usually "geometry"
    - coord_type: 'x', or 'y'
    ...
    Valid geometry types 'Polygon', 'MultiPolygon'
    """

    # Parse the geometries and grab the coordinate
    geometry = row[geom]

    if geometry.geom_type=='Polygon':
        if coord_type == 'x':
            # Get the x coordinates of the exterior
            # Interior is more complex: xxx.interiors[0].coords.xy[0]
            return list( geometry.exterior.coords.xy[0] )
        elif coord_type == 'y':
            # Get the y coordinates of the exterior
            return list( geometry.exterior.coords.xy[1] )

    if geometry.geom_type=='MultiPolygon':
        all_xy = []
        for ea in geometry.geoms:
            if coord_type == 'x':
                all_xy.append(list( ea.exterior.coords.xy[0] ))
            elif coord_type == 'y':
                all_xy.append(list( ea.exterior.coords.xy[1] ))
        return all_xy

def tracepolygones(gdf, couleur):
    """
    Calcule les coordonnées polaires des contours géographiques et les trace.
    gdf : un tableau GeoDataFrame contenant des polygones uniquement.
    """
    for index, row in gdf.iterrows():
        longi = getPolyCoords(row, 'geometry', 'x')
        lati = getPolyCoords(row, 'geometry', 'y')
        longi = np.radians(longi)
        lati = np.radians(lati)
        longi = longi - long_rad  # prise en compte de la longitude du cadran
        th, r = projection(longi,lati)
        ax.plot(th, r, c=couleur, zorder=0)

def tracemultigones(gdf, couleur):
    """
    Calcule les coordonnées polaires des contours géographiques et les trace.
    gdf : un tableau GeoDataFrame contenant des multipolygones uniquement.
    """
    for index, row in gdf.iterrows():
        longi = getPolyCoords(row, 'geometry', 'x')
        lati = getPolyCoords(row, 'geometry', 'y')
        for N in range(len(longi)):
            xx = np.array(longi[N])
            yy = np.array(lati[N])
            listex = []
            listey = []
            for i in range(len(xx)):
                listex.append(xx[i])
                listey.append(yy[i])
            listex = np.radians(listex)
            listey = np.radians(listey)
            listex = listex - long_rad  # prise en compte de la longitude du cadran
            th, r = projection(listex, listey)
            ax.plot(th, r, c=couleur, zorder=0)

 # =========================================================
 #  création de la figure et du graphique
fig = plt.figure(
    # figsize=(29.7/inch,21/inch)
    figsize=(fig_width, fig_height)
    )

ax = plt.subplot(111, projection='polar')
ax.set_theta_zero_location("S")  # origine des angles en bas
ax.set(
       # xticks=[],
       # yticks=[],
        thetamin=-Acone/2,
        thetamax=Acone/2,
        ylim=(r1,r2),#------------------
        rorigin = -r1,#------------------------
       )
# ax.set_xticks(np.radians(np.arange(-90,90,15)))
ax.set_xticks([])
ax.set_yticks([])
ax.grid(None)

# ====================================================
#  programme principal :
    #  ------- contours du patron ouvert :---
ax.plot([Aconerad/2,Aconerad/2],[r1,r2], **style_decoupe)
ax.plot([-Aconerad/2,-Aconerad/2],[r1,r2], **style_decoupe)

# =============================================================================
    #  import des données cartographiques (lignes de côte et frontières)

dfcotes = gpd.read_file('ne_110m_land/ne_110m_land.shp')
dffrontieres = gpd.read_file('ne_110m_admin_0_countries/ne_110m_admin_0_countries.shp')

# pour dffrontieres, séparation des données géographiques en polygones et multipolygones car la syntaxe de traitement est différente.
dffrontierespoly = dffrontieres[dffrontieres.geom_type != 'MultiPolygon']
dffrontieresmulti = dffrontieres[dffrontieres.geom_type == 'MultiPolygon']

# ------------------lignes horaires :

for heure in np.arange(int(heuremin), int(heuremax)+1):
    Hrad = np.radians((12-heure) * 15)  # angle horaire
    decl = np.linspace(-epsrad, epsrad,500, endpoint=True)
    th, ray = projection(Hrad, decl)
    ax.plot(th, ray, **style_lignes_hor)  # tracé
    # chiffraisons des heures autour :
    if heure == 12:
        idx1, idx2 = 0, len(th)-1
    else:
        idx1, idx2 = notnanindex(th)
    if r1 < ray[idx1]*1.02 < r2 and -Aconerad/2 < th[idx1] < Aconerad/2 :
        #  graduations bord : 
        ax.text(th[idx1], ray[idx1]*1.01, str(heure), va='top', ha='center', c=couleur_heures, fontweight='bold', rotation=180)
    if r1 < ray[idx2]*0.99 < r2 and -Acone/2 < th[idx2] < Acone/2 :
        #  graduations centre : 
        ax.text(th[idx2], ray[idx2]*0.99, str(heure), va='bottom', ha='center', c=couleur_heures, fontweight='bold')

#  ------------ demi-heures ------
for heure in np.arange(int(heuremin)-0.5, int(heuremax)+1.5):
    Hrad = np.radians((heure-12) * 15)  # angle horaire
    decl = np.linspace(-epsrad, epsrad,500, endpoint=True)
    th, ray = projection(Hrad, decl)
    ax.plot(th, ray, **style_demies)

#  -----------------  arcs diurnes : --------------
for decl in np.radians([-eps, -20.15, -11.47, 0, 11.47, 20.15, eps]) :
    #  calcul de l'arc semi-diurne pour chaque déclinaison :
    Hlim = np.arccos(-np.tan(lat_rad) * np.tan(decl))
    H = np.linspace(-Hlim, Hlim,1000)
    th, ray = projection(H, decl)
    if decl==0:
        ax.plot(th, ray, **style_equinoxes, zorder=-1)
    else:
        ax.plot(th, ray, **style_arcs_diurnes, zorder=-1)


# =============================================================================
        #  tracé de la carte :

tracepolygones(dffrontierespoly, couleur_carte)
tracemultigones(dffrontieresmulti, couleur_carte)
tracepolygones(dfcotes, couleur_carte)


# -------------- inscriptions------------
coordgeo = f"lat. {str(lat).replace('.',',')}°\nlon. {str(longitude).replace('.',',')}°"

thetaleg, distleg = projection(np.radians(lonleg-longitude), np.radians(latleg))
# ax.text(0.5, 0.72, "cadran solaire conique\n"+coordgeo,
#         ha='center', va='center', fontsize=10,
#         transform= ax.transAxes)

ax.text(thetaleg, distleg, "cadran solaire\nconique\n"+coordgeo,
        ha='center', va='center', fontsize=8)

# --------------------------------------------------------

# Ajustement des dimensions de l'axe, nécessaire pour que le rayon
# du tracé pour que le tracé final ait exactement la taille voulue
# dans le PDF généré.
# """
fig.subplots_adjust(
    left = marge_G / fig_width,
    right = -1.3-marge_D / fig_width,# !!!! à ajuster par tâtonnements !
    bottom = marge_B / fig_height,
    top = 1 - marge_H / fig_height,
)
# =============================================================================
fig.savefig("cadran_conique.png", dpi=300)
fig.savefig("cadran_conique.pdf", transparent=True)
