#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created Jul  2025
Author: David Alberto (www.astrolabe.science.fr)
Tracé des lignes d'un cadran solaire vertical méridional, destiné à être
placé au dos  d'une épaisseur de matériau transparent réfringent.
Une carte du monde est superposée au tracé ; le point d'ombre
indique le point subsolaire au fil de la journée.
Le pdf généré possède les dimensions exactes stipulées dans le script.
"""

import matplotlib.pyplot as plt
import numpy as np
import geopandas as gpd # gestion de données cartographiques

# =============================================================================
#-------------  réglages----------------------------
lat = 49.49  # latitude degrés (positive)
longitude = 0.107 # longitude du cadran solaire (négative si ouest)
n = 1.491  # indice de réfraction PMMA
# n = 1.333  # indice de réfraction eau
# n = 1.52  # indice de réfraction verre
# n = 1.591  # indice de réfraction polycarbonate
e = 5 # épaisseur de la couche réfringente
rayon = 5  # rayon du disque
couleur_heures = 'mediumblue'
style_lignes_hor = dict(lw=1, c=couleur_heures)
style_demies = dict(lw=0.6, c=couleur_heures,ls='--')
style_arcs_diurnes = dict(lw=0.6, c='yellowgreen')
style_equinoxes = dict(lw=0.8, c='darkolivegreen')
style_carte = dict(lw=0.4, c='sienna')
plt.rcParams["font.family"] = "Roboto Condensed"
plt.rcParams["font.size"] = 8
plt.rcParams["lines.solid_capstyle"] = "round"

# ################################################
lat_rad = np.radians(lat) # latitude radians
long_rad = np.radians(longitude)  # longitude radians
eps = 23 + 26/60  # obliquité de l'écliptique
epsrad = np.radians(eps)
Hmax = np.arccos(-np.tan(lat_rad) * np.tan(epsrad))  # angle horaire max
# heuremin = (12-np.degrees(Hmax)//15)  # heure minimale
heuremin = 7
# heuremax = (12+np.degrees(Hmax)//15)  # heure maximale
heuremax = 17
#  distance maximale d'une ligne :
rmax = e * np.tan(np.arcsin(1/n))
NomFichier = "cadran_vertical_carte"

# gestion des dimensions du graphique :
inch = 2.54 # conversion des cm en pouces
fig_width = (2 * rayon + 1) / inch 
fig_height = (2 * rayon + 1) / inch
marge_G = 0.5 / inch
largeur_axe = (2 * rayon) / inch
hauteur_axe = (2 * rayon) / inch
marge_D = fig_width-marge_G-largeur_axe
marge_B = marge_G
marge_H = fig_height-marge_B-hauteur_axe
# ==============================================================
# FONCTIONS
def azimut(H, decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie l'azimut du Soleil, en radians.
    """
    A = np.arctan2(np.cos(decl) * np.sin(H), np.sin(lat_rad) * np.cos(decl) * np.cos(H) - np.cos(lat_rad) * np.sin(decl))
    return A

def hauteur(H, decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie la hauteur du Soleil, en radians.
    """
    h = np.arcsin(np.sin(decl) * np.sin(lat_rad) + np.cos(decl) * np.cos(lat_rad) * np.cos(H))
    return h

def anglerefr(H, decl):
    """
    H : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie l'angle réfracté du rayon lumineux (radians).
    """
    h = hauteur(H, decl)
    A = azimut(H, decl)
    i = np.arccos(np.cos(A) * np.cos(h))
    ip = np.arcsin(np.sin(i)/n)
    return ip

def coordM(H,decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie les coordonnées polaires du point d'ombre sur le tracé.
    Theta en radians.
    """
    A = azimut(H, decl)
    #  correction de l'azimut : supprimer les données si abs(A) > 90° :
    A = np.where(abs(A)>np.pi/2,np.nan,A)
    h = hauteur(H, decl)
    #  suppression des hauteurs négatives (Soleil couché):
    h = np.where(h<0, np.nan, h)
    r = e * np.tan(anglerefr(H, decl))
    beta = np.arctan2(np.tan(h), np.sin(A))
    theta = beta  + np.pi
    return theta, r

def notnanindex(array):
    """
    array : array numpy.
    renvoie les index de la première et de la 
    dernière valeur qui ne soit pas un NaN.
    """
    if len(array[np.isfinite(array)]) >0:
        first = array[np.isfinite(array)][0]
        last = array[np.isfinite(array)][-1]
        idx1 = np.where(array==first)[0][0]
        idx2 = np.where(array==last)[0][0]
    else:
        idx1 = 0
        idx2 = 0
    return idx1, idx2

def getPolyCoords(row, geom, coord_type):
    """
    Returns the coordinates ('x|y') of edges/vertices of a Polygon/others
    Args:
    - row: the row object from a geodataframe; i.e.   df.loc[1]
    - geom: the name of "geometry" column, usually "geometry"
    - coord_type: 'x', or 'y'
    """

    # Parse the geometries and grab the coordinate
    geometry = row[geom]
    if geometry.geom_type=='Polygon':
        if coord_type == 'x':
            # Get the x coordinates of the exterior
            return list( geometry.exterior.coords.xy[0] )
        elif coord_type == 'y':
            # Get the y coordinates of the exterior
            return list( geometry.exterior.coords.xy[1] )

    if geometry.geom_type=='MultiPolygon':
        all_xy = []
        for ea in geometry.geoms:
            if coord_type == 'x':
                all_xy.append(list( ea.exterior.coords.xy[0] ))
            elif coord_type == 'y':
                all_xy.append(list( ea.exterior.coords.xy[1] ))
        return all_xy


def tracepolygones(gdf, couleur):
    """
    Calcule les coordonnées des points géographiques selon la 
    projection du cadran, puis trace les contours géographiques.
    Données en format polygones seulement.
    """
    for index, row in gdf.iterrows():
        longi = getPolyCoords(row, 'geometry', 'x')
        lati = getPolyCoords(row, 'geometry', 'y')
        longi = np.radians(longi)
        lati = np.radians(lati)
        longi = - longi
        # lati = - lati
        longi = longi + long_rad  # prise en compte de la longitude du cadran
        th, r = coordM(longi, lati)
        ax.plot(th, r, **style_carte)

def tracemultigones(gdf, couleur):
    """
    Calcule les coordonnées des points géographiques selon la 
    projection du cadran, puis trace les contours géographiques.
    Données en format multipolygones seulement.
    """
    for index, row in gdf.iterrows():
        x = getPolyCoords(row, 'geometry', 'x')
        y = getPolyCoords(row, 'geometry', 'y')
        for N in range(len(x)):
            xx = np.array(x[N])
            yy = np.array(y[N])
            listex = []
            listey = []
            for i in range(len(xx)):
                listex.append(xx[i])
                listey.append(yy[i])
            listex = np.radians(listex)
            listey = np.radians(listey)
            listex = - listex
            # listey = - listey
            listex = listex + long_rad  # prise en compte de la longitude du cadran
            th, r = coordM(listex, listey)
            ax.plot(th, r, **style_carte)


 # =============================================================================
 #  création de la figure et du graphique
fig = plt.figure(figsize=(fig_width, fig_height))

ax = plt.subplot(111, projection='polar')
ax.set(
       xticks=[],
       yticks=[],
       ylim=(0,rayon)
       )

ax.scatter(0,0,marker='+', lw=0.3)

# ===================================================================
#  programme principal :
    
    #  import des données cartographiques :
        
df = gpd.read_file('../ne_110m_land/ne_110m_land.shp')
dfc = gpd.read_file('../ne_110m_admin_0_countries/ne_110m_admin_0_countries.shp')

dfcpoly = dfc[dfc.geom_type != 'MultiPolygon']
dfcmulti = dfc[dfc.geom_type == 'MultiPolygon']
    
    #---------tracé de la carte :----------------------

tracepolygones(dfcpoly, 'peru')
tracemultigones(dfcmulti, 'peru')
tracepolygones(df, 'peru')
    
    # ----------- tracé du cadran solaire --------
    
    # ------------------lignes horaires :

for heure in np.arange(int(heuremin), int(heuremax)+1):
    Hrad = np.radians((heure-12) * 15)  # angle horaire
    decl = np.linspace(-epsrad, epsrad,500, endpoint=True)
    th, r = coordM(Hrad, decl)
    ax.plot(th, r, **style_lignes_hor)  # tracé
    # chiffraisons des heuures au bord :
    if heure == 12:
        idx1, idx2 = 0, len(th)-1
    else:
        idx1, idx2 = notnanindex(th)
    if r[idx2] + 0.05 < rayon:
        ax.text(th[idx2], r[idx2]*1.04, str(heure), va='top', ha='center', c=couleur_heures)
        # chiffraison sur l'intérieur :
    # ax.text(th[idx1], r[idx1]-0.12, str(heure), va='center', ha='center', c=couleur_heures)

    #  ------------ demi-heures ------
for heure in np.arange(int(heuremin)-0.5, int(heuremax)+1.5):
    Hrad = np.radians((heure-12) * 15)  # angle horaire
    decl = np.linspace(-epsrad, epsrad,500, endpoint=True)
    th, r = coordM(Hrad, decl)
    ax.plot(th, r, **style_demies)

    #  -----------------  arcs diurnes : --------------œ
for decl in np.radians([-eps, -20.15, -11.47, 0, 11.47, 20.15, eps]) :
    # calcul des arcs semi-diurnes :
    Hlim = np.arccos(-np.tan(lat_rad) * np.tan(decl))
    H = np.linspace(-Hlim, Hlim,1000)
    th, r = coordM(H, decl)
    #  tracé :
    if decl==0:
        ax.plot(th, r, **style_equinoxes, zorder=-1)
    else:
        ax.plot(th, r, **style_arcs_diurnes, zorder=-1)

#  ------------- lignes horizontales :---------------
ax.plot([0,0],[0.5*rayon,rmax], c='gray', lw=0.6)
ax.plot([np.pi,np.pi],[0.5*rayon,rmax], c='gray', lw=0.6)

#  ------------- lignes verticales :---------------
ax.plot([np.pi/2,np.pi/2],[0.3*rayon,0.6*rayon], c='gray', lw=0.6)
# ax.plot([-np.pi/2,-np.pi/2],[0.9*rayon,0.96*rayon], c='gray', lw=0.6)

#  points 6h et 18h :
ax.scatter(np.pi, rmax, s=2, c=couleur_heures, zorder=2)
if rmax < rayon:
    ax.text(np.pi, rmax+0.05, 18, c=couleur_heures, zorder=2,
        va='top', ha='right')
# 
ax.scatter(0, rmax, s=2, c=couleur_heures, zorder=2)
if rmax < rayon:
    ax.text(0, rmax+0.05, 6, c=couleur_heures, zorder=2,
        va='top', ha='left')


#  tracé d'un cercle de rayon rmax :
theta = np.linspace(0,2*np.pi, 1000, endpoint=True)
r = np.ones_like(theta) * rmax
ax.plot(theta, r, '-', c='gray', lw=0.4, zorder=0)    

# -------------- inscriptions------------
    
fig.text(0.05, 0.05, "cadran vertical\nméridional à réfraction",
        ha='left', va='bottom', fontsize=10, transform=fig.transFigure)

texte = f"lat.{str(lat).replace('.',',')}°N, lon.{longitude}°\nn={str(n).replace('.',',')}  e={str(e).replace('.',',')} cm"
fig.text(0.95,0.05,texte, c='indigo', ha='right', va='bottom',
        fontsize=7, transform=fig.transFigure
        # fontweight='bold'
        )

# --------------------------------------------------------
"""
Ajustement des dimensions de l'axe, nécessaire pour que le rayon
du tracé pour que le tracé final ait exactement la taille voulue
dans le PDF généré.
"""
fig.subplots_adjust(
    left = marge_G / fig_width,
    right = 1-marge_D / fig_width,
    bottom = marge_B / fig_height,
    top= 1 - marge_H / fig_height,
)
# =============================================================================
# fig.savefig(NomFichier + ".png", dpi=300)
fig.savefig(NomFichier + ".pdf")
