#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created Jul  2025
Author: David Alberto (www.astrolabe.science.fr)
Tracé des lignes d'un cadran solaire vertical méridional, destiné à être
placé au dos  d'une épaisseur de matériau transparent réfringent.
Le pdf généré possède les dimensions exactes stipulées dans le script.
"""

import matplotlib.pyplot as plt
import numpy as np

# =============================================================================
#-------------  réglages----------------------------
lat = 40  # latitude degrés
n = 1.491  # indice de réfraction PMMA
# n = 1.333  # indice de réfraction eau
# n = 1.591  # indice de réfraction polycarbonate
e = 6.0 # épaisseur de la couche réfringente
rayon = 6.0  # rayon du disque
couleur_heures = 'mediumblue'
style_lignes_hor = dict(lw=1, c=couleur_heures)
style_demies = dict(lw=0.6, c=couleur_heures,ls='--')
style_arcs_diurnes = dict(lw=0.6, c='yellowgreen')
style_equinoxes = dict(lw=0.8, c='darkolivegreen')
plt.rcParams["font.family"] = "Roboto Condensed"
plt.rcParams["font.size"] = 8
plt.rcParams["lines.solid_capstyle"] = "round"
plt.rcParams["lines.linewidth"] = 0.5


# ################################################
lat_rad = np.radians(lat)  # latitude radians
eps = 23 + 26/60  # obliquité de l'écliptique
epsrad = np.radians(eps)
Hmax = np.arccos(-np.tan(lat_rad) * np.tan(epsrad))  # angle horaire max
# heuremin = (12-np.degrees(Hmax)//15)  # heure minimale
heuremin = 7
# heuremax = (12+np.degrees(Hmax)//15)  # heure maximale
heuremax = 17
#  distance maximale d'une ligne :
rmax = e * np.tan(np.arcsin(1/n))
NomFichier = f"cadran_horiz-{str(lat)}"

# gestion des dimensions du graphique :
inch = 2.54 # conversion des cm en pouces
fig_width = (2 * rayon + 1) / inch 
fig_height = (2 * rayon + 1) / inch
marge_G = 0.5 / inch
largeur_axe = (2 * rayon) / inch
hauteur_axe = (2 * rayon) / inch
marge_D = fig_width-marge_G-largeur_axe
marge_B = marge_G
marge_H = fig_height-marge_B-hauteur_axe
# ==============================================================
# FONCTIONS
def azimut(H, decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie l'azimut du Soleil, en radians.
    """
    A = np.arctan2(np.cos(decl) * np.sin(H), np.sin(lat_rad) * np.cos(decl) * np.cos(H) - np.cos(lat_rad) * np.sin(decl))
    return A

def hauteur(H, decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie la hauteur du Soleil, en radians.
    """
    h = np.arcsin(np.sin(decl) * np.sin(lat_rad) + np.cos(decl) * np.cos(lat_rad) * np.cos(H))
    return h

def anglerefr(H, decl):
    """
    H : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie l'angle de réfraction du rayon lumineux.
    Renvoie l'angle réfracté du rayon solaire, en radians.
    """
    h = hauteur(H, decl)
    A = azimut(H, decl)
    i = np.arccos(np.cos(A) * np.cos(h))
    ip = np.arcsin(np.sin(i)/n)
    return ip

def coordM(H,decl):
    """
    H, decl : float.
    H : angle horaire du Soleil (radians)
    decl : déclinaison du Soleil (radians)
    Renvoie les coordonnées polaires du point d'ombre. Theta en radians.
    """
    A = azimut(H, decl)
    #  correction de l'azimut : supprimer les données si abs(A) > 90° :
    A = np.where(abs(A)>np.pi/2,np.nan,A)
    h = hauteur(H, decl)
    #  suppression des données de hauteur négative :
    h = np.where(h<0, np.nan, h)
    r = e * np.tan(anglerefr(H, decl))
    beta = np.arctan2(np.tan(h), np.sin(A))
    theta =  beta 
    theta = theta + np.pi ## origine des angles à droite
    return theta, r

def notnanindex(array):
    """
    array : array numpy.
    renvoie les index de la première et de la 
    denière valeur qui ne soit pas un NaN.
    """
    if len(array[np.isfinite(array)]) >0:
        first = array[np.isfinite(array)][0]
        last = array[np.isfinite(array)][-1]
        idx1 = np.where(array==first)[0][0]
        idx2 = np.where(array==last)[0][0]
    else:
        idx1 = 0
        idx2 = 0
    return idx1, idx2

 # =========================================================
 #  création de la figure et du graphique
fig = plt.figure(figsize=(fig_width, fig_height))

ax = plt.subplot(111, projection='polar')
ax.set(
       xticks=[],
       yticks=[],
       ylim=(0,rayon)
       )

ax.scatter(0,0,marker='+', lw=0.3) # croix au centre

# ====================================================
#  programme principal :
   
# ------------------lignes horaires :

for heure in np.arange(int(heuremin), int(heuremax)+1):
    Hrad = np.radians((heure-12) * 15)  # angle horaire
    decl = np.linspace(-epsrad, epsrad,500, endpoint=True)
    th, r = coordM(Hrad, decl)
    ax.plot(th, r, **style_lignes_hor)  # tracé
    # chiffraisons des heures autour :
    if heure == 12:
        idx1, idx2 = 0, len(th)-1
    else:
        idx1, idx2 = notnanindex(th)
    # ax.text(th[idx1], r[idx1]-0.12, str(heure), va='center', ha='center', c='royalblue')
    if r[idx2] + 0.05 < rayon:
        ax.text(th[idx2], r[idx2]*1.03, str(heure), va='top', ha='center', 
                c='midnightblue')

#  ------------ demi-heures ------
for heure in np.arange(int(heuremin)-0.5, int(heuremax)+1.5):
    Hrad = np.radians((heure-12) * 15)  # angle horaire
    decl = np.linspace(-epsrad, epsrad,500, endpoint=True)
    th, r = coordM(Hrad, decl)
    ax.plot(th, r, **style_demies)

#  -----------------  arcs diurnes : --------------
for decl in np.radians([-eps, -20.15, -11.47, 0, 11.47, 20.15, eps]) :
    #  calcul de l'arc semi-diurne pour chaque déclinaison :
    Hlim = np.arccos(-np.tan(lat_rad) * np.tan(decl))
    H = np.linspace(-Hlim, Hlim,1000)
    th, r = coordM(H, decl)
    if decl==0:
        ax.plot(th, r, **style_equinoxes, zorder=-1)
    else:
        ax.plot(th, r, **style_arcs_diurnes, zorder=-1)

#  ------------- lignes horizontales :---------------
ax.plot([0,0],[0.5*rayon,rmax], c='gray', lw=0.6)
ax.plot([np.pi,np.pi],[0.5*rayon,rmax], c='gray', lw=0.6)

#  ------------- lignes verticales :---------------
ax.plot([np.pi/2,np.pi/2],[0.3*rayon,0.6*rayon], c='gray', lw=0.6)
# ax.plot([-np.pi/2,-np.pi/2],[0.9*rayon,0.96*rayon], c='gray', lw=0.6)

#  points 6h et 18h :
ax.scatter(np.pi, rmax, s=2, c='midnightblue', zorder=2)
if rmax < rayon:
    ax.text(np.pi, rmax*1.02, 18, c='midnightblue', zorder=2,
        va='top', ha='right')
# 
ax.scatter(0, rmax, s=2, c='midnightblue', zorder=2)
if rmax < rayon:
    ax.text(0, rmax*1.02, 6, c='midnightblue', zorder=2,
        va='top', ha='left')

# -------------- inscriptions------------
texte = f"{str(lat).replace('.',',')}°N\nn={str(n).replace('.',',')}\ne={str(e).replace('.',',')} cm"
ax.text(np.pi/2,0.25,texte, c='indigo', ha='center', va='bottom',
        fontsize=10,
        # fontweight='bold'
        )
    
fig.text(0.05, 0.05, "cadran vertical\nméridional\nà réfraction",
        ha='left', va='bottom', fontsize=10,
        transform= fig.transFigure)
    

#  tracé d'un cercle de rayon rmax :
theta = np.linspace(0,2*np.pi, 1000, endpoint=True)
r = np.ones_like(theta) * rmax
ax.plot(theta, r, '-', c='gray', lw=0.4, zorder=0)    

# --------------------------------------------------------
"""
Ajustement des dimensions de l'axe, nécessaire pour que le rayon
du tracé pour que le tracé final ait exactement la taille voulue
dans le PDF généré.
"""
fig.subplots_adjust(
    left = marge_G / fig_width,
    right = 1-marge_D / fig_width,
    bottom = marge_B / fig_height,
    top= 1 - marge_H / fig_height,
)
# =============================================================================
fig.savefig("cadran_refraction_vertical.png", dpi=300)
fig.savefig("cadran_refraction_vertical.pdf")
