# -*- coding: utf-8 -*-
"""
Créé en avril 2023
auteur : David ALBERTO (www.astrolabe-science.fr)
"""
import matplotlib.pyplot as plt
import numpy as np
import locale # règles locales (noms des mois)
from skyfield.api import load
from skyfield.framelib import ecliptic_frame
import imageio.v2 as imageio


# --------------------------------- REGLAGES UTILISATEUR :
plt.rcParams["font.family"] = "Roboto"              # ou autre police installée sur l'ordinateur
pasJours=10                                          # intervalle de jours entre deux dates
locale.setlocale(locale.LC_ALL, 'fr_FR.UTF-8')      # dates en français
couleurFond='#c7cef2'                               # couleur d'arrière-plan de la figure
taillePoint=40                                      # taille du point des planètes sur le tracé
ts = load.timescale()#crée une échelle de dates Skyfield
periode = ts.utc(2024, 11, range(1, 150, pasJours)) # année, n° mois, série de jours, qui peut meme excéder 30 jours
# periode = ts.utc(2023, 4, range(1, 365, 15)) # exemple pour 1 an, tous les 15 jours
NomFichier = "PlanetesHelio"
rayonFleche = 1 # flèche indiquant la direction des planètes hors champ
# ---------------------------------

#Liste des couleurs par planète :
couleurs={
'Mercure':'gray',
'Vénus':'tan',
'Terre':'royalblue', 
'Mars':'orangered',
'Jupiter':'orange',
'Saturne':'darkgoldenrod',
'Uranus' : 'skyblue', 
'Neptune' : 'mediumblue'
}

# FIN DES REGLAGES UTILISATEURS

#%%%%%%%%%%%%%%%%%%% Fonctions
def coordonnees(planete, date): # calcule et renvoie la longitude écliptique (degrés) et la distance (UA)
    """
    planete : astre défini par Skyfield
    date : date ou série de dates Skyfield
    renvoie la longitude écliptique en radians, et la distance au Soleil en UA
    """
    lat, lon, distance = soleil.at(date).observe(planete).frame_latlon(ecliptic_frame)
    lon = lon.radians
    # lon = angleComplem(lon) # si discontinuité de la longitude pendant la période
    distance = distance.au
    return lon, distance

# def angleComplem(serie):
#     for n,  angle in enumerate(serie):
#         if angle >np.pi:
#             angle=angle-2*np.pi
#         serie[n]=angle
#     return serie

def date_JMA(date): # renvoie une date Skyfield au format Jour Mois Année (string)
    return date.utc_strftime('%d %b %Y')

def trajectoire(planete, date):
    """
    planete : astre défini par Skyfield
    date : date ou série de dates définies par Skyfield
    Trace les positions de la planète sur le graphique.
    """
    long, dist = coordonnees(planete, date)
    couleur, nom = attributs_planete(planete)
    # ax.yticks([], [])#suppression des cercles et graduations en rayon.
    # ax.set_yticks([1], [])#conserver un cercle de rayon 1 UA.
    ax.set_axisbelow(True)#place la grille sous les points tracés.
    # ax.set_yticks(np.arange(0, 2, 1), [])#graduer tous les 0.5 UA,sans chiffraison
    # ax.set_xticks(np.radians(np.arange(0, 360, 15)))#lignes d'angles, tous les 15°
    ax.scatter(long,dist,s=taillePoint, color=couleur, label=nom)
    ax.set_yticks(np.arange(0, 2, 1), [])#graduer tous les 0.5 UA,sans chiffraison
    ax.set_ylim(0, 2)
    ax.set_xticks(np.radians(np.arange(0, 360, 15)))#lignes d'angles, tous les 15°
    if date == periode[0]:
        ax.legend()
    # ax.scatter(long[0],dist[0],s=taillePoint, color='black') # 1e position en noir

def attributs_planete(planete):
    """
    planete : astre défini par Skyfield
    Renvoie la couleur et le nom (string) de l'astre
    RANG est le rang de la planète depuis le Soleil
    """
    if planete==mercure:
        rang = 0
    elif planete == venus:
        rang = 1
    elif planete == terre:
        rang = 2
    elif planete == mars:
        rang = 3
    elif planete == jupiter:
        rang = 4
    elif planete == saturne:
        rang = 5
    elif planete == uranus:
        rang = 6
    elif planete == neptune:
        rang = 7
    couleur= list(couleurs.values())[rang]
    nom= list(couleurs.keys())[rang]
    return couleur, nom

def indicateur_planete(planete,date):
    """
    planete : astre défini par Skyfield
    date : periode définie par Skyfield
    Trace une flèche dirigée vers la position moyenne de la planète sur la période considérée.
    La flèche est légendée avec le nom de la planète.
    Cette représentation évite des problèmes d'échelle pour les planètes très lointaines.
    En supposant que l'époque considérée soit suffisamment courte pour que la longitude de ces planetes
    varie très peu.
    Attention : si durant l'époque considérée, la longitude de la planète prend des valeurs de part et d'autre de 0°,
    la discontinuité de la longitude va donner une moyenne fantaisiste.
    La fonction AngleComplémentaire() permet de contourner le problème, le cas échéant.
    """
    lon, dist = coordonnees(planete, date)
    long_m= np.mean(lon)
    couleur, nom = attributs_planete(planete)
    ax.annotate("", xy=(long_m, rayonFleche), xytext=(0, 0),arrowprops=dict(arrowstyle="->", color=couleur), zorder=3)
    ax.text(long_m, rayonFleche+0.1, nom, ha='left',va='center', rotation=np.degrees(long_m), color=couleur)

def opposition(planete,dates):
    """
    Trace des segments entre la position de la Terre et celle de la planète, pour illustrer la rétrogradation
    """
    L_Terre, D_Terre =  coordonnees(terre, dates)
    L_planete, D_planete = coordonnees(planete, dates)
    ax.plot((L_Terre, L_planete), (D_Terre, D_planete), 'k')

#%%%%%%%%%%
# (Télé)chargement du fichier d'ephémérides de421.bsp (16 MB, période 1849-2050)
eph = load('../../../skyfield-ephemerides/de421.bsp')# si le fichier est déjà chargé sur le disque (chemin à adapter).
# eph=api.load('de421.bsp')# pour télécharger automatiquement le fichier, dans le dossier du fichier Python

soleil = eph['sun']
mercure = eph['mercury']
venus = eph['venus']
terre= eph['earth']
mars = eph['mars']
jupiter = eph['jupiter barycenter']
saturne = eph['saturn barycenter']
uranus = eph['uranus barycenter']
neptune = eph['neptune barycenter']

#Paramètres du graphique : --------------------------------------

#fig=plt.figure(figsize=(21/2.54, 29.7/2.54 ), tight_layout=True) # dimensions A4
fig=plt.figure(figsize=(10, 10), tight_layout=True, facecolor=couleurFond)
ax=plt.subplot(projection='polar')
plt.title("Rétrogradation de Mars", pad=10, fontsize=16, fontweight='bold')
plt.scatter(0, 0, s=200, c='orange', zorder=2)    #Soleil



filenames=[]
for N, date in enumerate(periode):
    for planete in [terre, mars]:
        trajectoire(planete, date)
    opposition(mars, date)
    filename = NomFichier + f"{N}.png"
    filenames.append(filename)
    #Affichage des dates :
    texte=date_JMA(date)
    ax.text(0.12, 0.90, texte, transform=fig.transFigure,
             c='white', va='top', fontsize=12, bbox=dict(fc='orangered'))
    fig.savefig(filename, dpi=50)
#----------------------------------------------------------------------------

with imageio.get_writer('PlanetesHelio.gif', mode='I', fps=1.5) as writer:
    for filename in filenames:
        image = imageio.imread(filename)
        writer.append_data(image)


# signature
fig.text(0.99,0.02,'David ALBERTO (www.astrolabe-science.fr)',rotation=90,
        ha='right',transform=fig.transFigure)

# plt.show()
