#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Crée le Mon Feb 14 10:34:29 2022
@auteur: david ALBERTO      (www.astrolabe-science.fr)
Ce code utilise le module mendeleev (données sur les éléments chimiques)
Dans ce module, un élément est appelé par element(N) pour récupérer
les données suiantes :
    - symbole
    - period : c'est le numéro de ligne
"""
# Appel des modules
import matplotlib.pyplot as plt
import matplotlib as mpl#Normalisation des valeurs
import pandas as pd
import numpy as np# gestion des données numériques absentes (NaN)
from mendeleev import element
import matplotlib.colorbar as cbar#barres de couleurs
from mpl_toolkits.axes_grid1 import make_axes_locatable#positionner les barres de couleurs

 # Paramètres personnalisables-----

"""
Les paramètres suivants sont liés entre eux, pour le rendu final.
Ils doivent être ajustés manuellement :
    - dimensions de la figure
    - valeurs des paramètres d'échelle a et b
    - taille des caractères
    - les différents offsets en x et en y
"""
couleur_fond = 'lightcyan'
Zmax = 118

# sélection des numéros atomiques à afficher :
# liste_Z = np.concatenate((
#     np.arange(1,56+1),
#     np.arange(71,86)
#     ))
liste_Z = np.arange(1,Zmax+1) # pour tous les éléments

#facteurs d'échelle :
a=0.22 #en abscisses
b=0.28 #en ordonnées. NB : les ordonnées sont négatives

# Réglages d'écriture du symbole des éléments :
taille_symbole=16
c_symbole='k'#couleur symbole

# Réglages d'écriture du numéro atomique :
x_offset_Z=-0.40
y_offset_Z=0.30
taille_Z=8
c_Z='k'#couleur numéro atomique

# Réglages d'écriture des terres rares :
y_offset_La=2.5#décalage vertical lanthanides

# Réglages d'écriture du nom des éléments :
x_offset_nom=0.0
y_offset_nom=-0.4
c_nom='black'
taille_nom=6

plt.rcParams["font.family"] = "Roboto" # ou autre police installée
# ---------------------------------


def existe(fichier):
    """
    Renvoie True si le fichier existe dans le répertoire du script, False sinon.
    """
    try:
        f = open(fichier,'r')
        f.close()
        return True
    except:
        return False

def couleur(Z):
    """
    À partir du numéro atomique Z, renvoie la couleur de la case, selon
    l'abondance et le colormap.
    """
    abond = abondance_log[Z-1]
    if abond != np.nan:
        couleur = cmap(norm(abond))
    else:
        couleur = 'gray'
    return couleur

def colonnes(Z):
    """
    renvoie la colonne d'un élément.
    """
    if Z <= 18:
        col = element(Z).group_id
    elif Z >= 57 and Z <= 70:#lanthanides
        col = Z - 54
    elif Z >= 89 and Z <= 102:# actinides
        col = Z - 86
    elif Z >= 71 and Z <= 86:
        col = Z - 68
    elif Z >= 103:
        col = Z - 100
    elif Z % 18 == 0:
        col = 18
    elif Z <= 86:
        col = Z % 18
    elif Z >= 87:
        col = Z - 86
    return col

def lignes(Z):
    """
    renvoie la ligne d'un élément.
    """
    if Z <= 18:
        lig = element(Z).period
    elif Z >= 57 and Z <= 70:#lanthanides
        lig = element(Z).period + y_offset_La
    elif Z >= 89 and Z <= 102:# actinides
        lig = element(Z).period + y_offset_La
    elif Z >= 71 and Z <= 86:
        lig = element(Z).period
    elif Z >= 103:
        lig = element(Z).period
    elif Z % 18 == 0:
        lig = Z // 18 + 2
    elif Z <= 86:
        lig = Z // 18 + 3
    elif Z >= 87:
        lig = element(Z).period
    return lig


#---------------------------------------------------------
noms=['Hydrogène','Hélium','Lithium','Béryllium','Bore','Carbone','Azote','Oxygène',
      'Fluor','Néon','Sodium','Magnésium','Aluminium','Silicium','Phosphore','Soufre',
      'Chlore','Argon','Potassium','Calcium','Scandium','Titane','Vanadium','Chrome',
      'Manganèse','Fer','Cobalt','Nickel','Cuivre','Zinc','Gallium','Germanium',
      'Arsenic','Sélénium','Brome','Krypton','Rubidium','Strontium','Yttrium',
      'Zirconium','Niobium','Molybdène','Technétium','Ruthénium','Rhodium',
      'Palladium','Argent','Cadmium','Indium',
'Étain','Antimoine','Tellure','Iode','Xénon','Césium','Baryum','Lanthane','Cérium',
'Praséodyme','Néodyme','Prométhium','Samarium','Europium','Gadolinium','Terbium',
'Dysprosium','Holmium','Erbium','Thulium','Ytterbium','Lutetium','Hafnium','Tantale',
'Tungstène','Rhénium','Osmium','Iridium','Platine','Or','Mercure','Thallium','Plomb',
'Bismuth','Polonium','Astate','Radon','Francium','Radium','Actinium','Thorium','Protactinium',
'Uranium','Neptunium','Plutonium','Americium','Curium','Berkelium','Californium','Einsteinium',
'Fermium','Mendelevium','Nobelium','Lawrencium','Rutherfordium','Dubnium','Seaborgium','Bohrium',
'Hassium','Meitnerium','Darmstadtium','Roentgenium','Copernicium','Nihonium','Flerovium',
'Moscovium','Livermorium','Tennesse','Oganesson',
]#le module mendeleev n'a que les noms des éléments en anglais
# ---------------------------------

# import des données d'abondance dans le système solaire, d'après fichier externe :
abondanceSS = pd.read_csv('abondance_systeme_sol.csv', skiprows=1, sep='\t')


if existe('tableau.csv'):
    tableau = pd.read_csv("tableau.csv", sep='\t')
else:# creation des listes de lignes et colonnes, ainsi que du fichier csv les contenant.
    ligne=[]
    colonne=[]
    symbole=[]
    for Z in liste_Z:
        Z = int(Z)
        col = colonnes(Z)
        lig = lignes(Z)
        symb = element(Z).symbol
        colonne.append(col)
        ligne.append(lig)
        symbole.append(symb)

    #enregistrement pour usage ultérieur, afin de gagner du temps
    tableau = pd.DataFrame(
                    {'Z' : liste_Z,
                     'nom' : noms[0:len(liste_Z)],
                     'col' : colonne,
                     'lig' : ligne,
                     'symbole' : symbole,
                     'abondance': np.nan
                        }
        )

    # itération sur abondanceSS pour chercher les abondances et les placer dans 'tableau
    for Z in abondanceSS['Z'].values:
        abo = abondanceSS.loc[abondanceSS['Z'] == Z, 'abondance'].values[0]
        abo = np.log10(abo)
        tableau.loc[tableau['Z'] == Z, 'abondance'] = abo

    tableau.to_csv('tableau.csv',index=False, sep='\t', na_rep='nan')

abondance_log = tableau['abondance']
# ---------------------------------
#gamme de couleurs choisie (colormap) :
cmap=plt.get_cmap('viridis_r')
"""
# Le colormap est à choisir parmi les séquentiels :
'Greys', 'Purples', 'Blues', 'Greens', 'Oranges', 'Reds',
'YlOrBr', 'YlOrRd', 'OrRd', 'PuRd', 'RdPu', 'BuPu',
'GnBu', 'PuBu', 'YlGnBu', 'PuBuGn', 'BuGn', 'YlGn'
"""
#normalisation de la gamme de valeurs :
norm = mpl.colors.Normalize(vmin=min(abondance_log),vmax=max(abondance_log))

# paramètres du graphique
fig=plt.figure(figsize=(29.7/2.54,21/2.54),tight_layout=True,facecolor=couleur_fond)


# plt.rcParams.update({"text.usetex": True}) #utiliser la police LaTeX (optionnel)
ax=plt.subplot(111)

plt.title("Abondance des éléments chimiques dans le Système solaire",fontsize=20,y=0.90)
plt.xlim(0,19*a)
# plt.ylim(-7.0*b,0)
plt.ylim(-10.5*b,0) # si lanthanides, actinides
plt.xticks([])
plt.yticks([])
ax.set_aspect('equal')
# ---------------------------------

# Tracé du diagramme :
for Z in liste_Z:
    x = tableau.loc[tableau['Z']==Z, 'col'] * a
    y = - tableau.loc[tableau['Z']==Z,'lig'] * b
    nom = tableau.loc[tableau['Z']==Z,'nom'].values[0]
    pourcentage = tableau.loc[tableau['Z']==Z,'abondance'].values[0]
    if pourcentage >= 0:
        c_texte = 'white'
    elif np.isnan(pourcentage):
        c_texte = 'gray'
    else:c_texte = c_nom
    symbole = element(int(Z)).symbol
    plt.text(x,y,symbole,fontsize=taille_symbole,va='center',ha='center', c = c_texte) # symbole
    couleur_case = couleur(Z)
    rectangle = plt.Rectangle((x-0.5*a, y - 0.5*b),a,b,facecolor=couleur_case,edgecolor='white',linewidth=1)#case
    ax.add_patch(rectangle)
    if Z in [91,101,104,110,116]:#éléments aux noms longs
        tailleDuNom = taille_nom - 1
    else: tailleDuNom = taille_nom
    plt.text(x + x_offset_Z * a, y + y_offset_Z * b, Z, c = c_texte, fontsize = taille_Z, ha='left')#numéro atomique
    plt.text(x + x_offset_nom * a, y + y_offset_nom * b, nom, c = c_texte, fontsize = tailleDuNom,ha='center')#nom

# Trait pour les terres rares :
plt.plot([2.5*a,2.5*a],[-6.5*b,-8.0*b],'--',c='gray',lw=1)

# signature :
plt.text(0.985,0.02,'D. Alberto (www.astrolabe-science.fr) Données : La radioactivité, Y. Chelet, Nucleon 2006.',
         rotation=90,fontsize=6,transform=ax.transAxes)

# Numéros des colonnes :
# for col in range(1,19):
#     if col > 2 and col < 13:
#         ligne = -3.4 * b
#     elif col==1 or col== 18 :
#         ligne = -0.4 * b
#     else: ligne = -1.4 * b
#     ax.text(a * col,ligne,f'{col}',color='darkgray',fontsize=10,fontweight='bold',
#              ha='center')

#tracé de la barre de couleurs :
divider = make_axes_locatable(ax)
cax = divider.append_axes('bottom', size='6%', pad = 0.2)
cb=cbar.ColorbarBase(cax, cmap=cmap, norm=norm, orientation='horizontal')
cax.set_xlabel('log(%)')
cax.set_xlim(min(abondance_log),max(abondance_log))

NomFichier="TableauPerAbondanceSystSol"
fig.savefig(NomFichier+'.png',dpi=200)
fig.savefig(NomFichier+'.pdf')
