#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Créé le Wed May 31 22:07:04 2023

@auteur: David ALBERTO
(www.astrolabe-science.fr)
Ce script trace des spectres RMN, soit d'après des valeurs expérimentales
trouvées sur internet, soit des spectres simulés.
"""
import matplotlib.pyplot as plt # graphiques
import numpy as np # listes
import pandas as pd # tableaux de données et import/export


plt.rcParams["font.family"] = "Roboto" #ou autre police installée
plt.rcParams["font.size"] = 6

# molecule = 'ethylformiate'
molecule = 'ascorbic'
xmin, xmax = 3.5, 5.1 # à ajuster aux données

NomFichieBrut = molecule + '.csv'
NomFichierExport = molecule + '_continu.csv'

prop_ligne = dict(lw=0.5, c= 'royalblue')# mise en forme de la ligne de la courbe

spectre = pd.read_csv(molecule+'.csv', sep=',')# import des données

# spectre créé dans le script, plutot que dans un fichier externe :
# spectre = pd.DataFrame({'ppm': [2, 4, 6, 7],
#                         'Int.':[2, 2, 2, 2],
#                         'mult':['t', 'd', 's', 'q']
#                         }
#                         )

ppm = spectre['ppm']
if 'mult' in spectre.columns:
    mult = spectre['mult']
H = spectre['Int.']

delta = 0.2 # écart entre les pics, pour les multiplets modélisés
a = 80000 # inverse de la largeur des pics (+ grand, + fin)

lettres_mult = ['s','d','t','q','qt','sx','h','o','n','dc']# singulet, doublet, triplet, etc.

def multiplet(m, ppm, H):
    """
    Cette fonction est utilisée pour tracer automatiquement un multiplet
    à l'abscisse demandée, et d'intensité H.
    m : str (multiplicité)
    H : (entier) : intensité du signal (nombre de protons)
    ppm (flottant): abscisse du signal
    renvoie une liste de valeurs Y en ordonnées
    I est l'ordonnée du pic le + faible du multiplet
    """
    N = lettres_mult.index(m) + 1 # entier correspondant au multiplet
    # print(N)
    I = H / (2**(N-1)) # maximum du pic le plus petit du multiplet
    deltamin = 0.5*(1-N) # abscisse du 1er pic, par rapport au centre du signal
    f = Y
    for i in range(0,N):
        decalageX = delta * (deltamin + i) # decalage de chaque pic par rapport au centre
        coef = coef_pascal(N, i+1)
        f = f + I * coef * np.exp(-a*(X - ppm - decalageX)**2)
    return f

def pic_isole(ppm, H):
    """
    Cette fonction sert à tracer une courbe de Gauss à l'abscisse donnée ;
    elle est adaptée pour le tracé d'un spectre à partir de données exp.
    H : (entier) : intensité du signal (nombre de protons)
    ppm (flottant): abscisse du signal
    renvoie une liste de valeurs Y en ordonnées
    """
    Y = H * np.exp(-a*(X-ppm)**2)
    return Y

def coef_pascal(n, k):
    """
    retourne la valeur du cefficient du triangle de Pascal, de ligne n et de rang k.
    """
    C = 1
    for j in range(1,k):
            C = C * (n - j) // j
    return C

def export_fichier(X,Y):
    """
    crée un DataFrame puis un fichier csv pour exporter les valeurs calculées.
    """
    df = pd.DataFrame({'ppm': X,
                        'I': Y})
    df.to_csv(NomFichierExport, index=False, sep='\t')

fig, ax = plt.subplots()

ax.invert_xaxis()
# ax.set_xlim(xmax,xmin)
ax.set_xticks(np.arange(xmin,xmax,0.1),minor=True)
ax.set_xlabel(r'$\delta$ (ppm)')
ax.set_yticks([])

X = np.arange(xmin,xmax,0.003)
Y = np.zeros_like(X)

# ax.grid(lw=0.2)
# itération sur le tableau 'spectre' :----------------------
for signal in range(len(spectre)):
    x = ppm[signal]
    intensite = H[signal]
    if 'mult' in spectre.columns:
        m = mult[signal]
        # Y = Y + multiplet(m, x, intensite)
        Y = multiplet(m, x, intensite)
    else:
        Y = Y + pic_isole(x, intensite)
    # pic supplémentaire :
# x = 5
# Y = Y + pic_isole(x, 4)


ax.plot(X,Y, **prop_ligne)
ax.text(0.01, 0.99, molecule, transform=ax.transAxes,va='top',c='firebrick',
        fontsize=10)

export_fichier(X, Y)

# def integration(H):
#     """
#     H : nombre d'atomes H dans la molécule (pour normalisation).
#     renvoie integr : la courbe d'intégration normalisée (np.array)
#     """
#     ymin,ymax = ax.get_ylim()
#     integration_total = 0
#     integr = [Y[0]]
#     for N in range(1,len(X)):
#         aire = Y[N] * 1
#         integration_total += aire
#         integr.append(integr[N-1] + aire)
#     integr = np.array(integr) / max(integr) * ymax*0.8 # normalisation
#     return integr

# integr = integration(5)
# ax.plot(X,integr,c='C1', alpha=0.5, lw=0.8)


fig.savefig('spectreRMN' + molecule + '.png',dpi=200)
fig.savefig('spectreRMN' + molecule + '.pdf')
