# -*- coding: utf-8 -*-
"""
Créé en avril 2023
auteur : David ALBERTO (www.astrolabe-science.fr)
"""
import matplotlib.pyplot as plt
import numpy as np
import locale # règles locales (noms des mois)
from skyfield.api import load

# --------------------------------- REGLAGES UTILISATEUR :
plt.rcParams["font.family"] = "Roboto"              # ou autre police installée sur l'ordinateur
pasJours=2                                          # intervalle de jours entre deux dates
NJours=1000                                          # nombre de jours
locale.setlocale(locale.LC_ALL, 'fr_FR.UTF-8')      # dates en français
couleurFond='#c7cef2'                               # couleur d'arrière-plan de la figure
ts = load.timescale()#crée une échelle de dates Skyfield
periode = ts.utc(2023, 6, range(1, NJours, pasJours)) # année, n° mois, série de jours, qui peut meme excéder 30 jours
# periode = ts.utc(2023, 4, range(1, 365, 15)) # exemple pour 1 an, tous les 15 jours
NomFichier = "DistancesPlanetesTerre"
# CHOISIR LA PLANETE DE REFERENCE PLUS BAS DANS LE SCRIPT.
# PUIS, CHOISIR LES PLANETES TESTEES.
# ---------------------------------

#Liste des couleurs par planète :
couleurs={
'Mercure':'gray',
'Vénus':'tan',
'la Terre':'royalblue',
'Mars':'orangered',
'Jupiter':'orange',
'Saturne':'darkgoldenrod',
'Uranus' : 'skyblue', 
'Neptune' : 'mediumblue'
}

# FIN DES REGLAGES UTILISATEURS

date_debut = periode[0].utc_strftime()
date_fin = periode[-1].utc_strftime()
#%%%%%%%%%%%%%%%%%%% Fonctions

def date_JMA(date): # renvoie une date Skyfield au format Jour Mois Année (string)
    return date.utc_strftime('%d %b %Y')

def attributs_planete(planete):
    """
    planete : astre défini par Skyfield
    Renvoie la couleur et le nom (string) de l'astre
    RANG est le rang de la planète depuis le Soleil
    """
    if planete==mercure:
        rang = 0
    elif planete == venus:
        rang = 1
    elif planete == terre:
        rang = 2
    elif planete == mars:
        rang = 3
    elif planete == jupiter:
        rang = 4
    elif planete == saturne:
        rang = 5
    elif planete == uranus:
        rang = 6
    elif planete == neptune:
        rang = 7
    couleur= list(couleurs.values())[rang]
    nom= list(couleurs.keys())[rang]
    return couleur, nom

def distance(planete1,planete2,date):
    """
    renvoie la distance entre deux planètes, à la date donnée.
    """
    P1 = planete1.at(date)
    P2 = planete2.at(date)
    dist = (P2-P1).distance().au # distance en UA
    return dist

#%%%%%%%%%%
# (Télé)chargement du fichier d'ephémérides de421.bsp (16 MB, période 1849-2050). Il existe d'autres fichiers, voir documentation Skyfield
eph = load('../skyfield-ephemerides/de421.bsp')# si le fichier est déjà chargé sur le disque (chemin à adapter).
# eph=api.load('de421.bsp')# pour télécharger automatiquement le fichier, dans le dossier du fichier Python

soleil = eph['sun']
mercure = eph['mercury']
venus = eph['venus']
terre= eph['earth']
mars = eph['mars']
jupiter = eph['jupiter barycenter']
saturne = eph['saturn barycenter']
uranus = eph['uranus barycenter']
neptune = eph['neptune barycenter']

# CHOIX DE LA PLANETE DE REFERENCE (nom sans majuscule ni accent):-----------------------------
planete_reference = terre
couleur_ref, nom_ref = attributs_planete(planete_reference)

# CHOIX DES PLANETES A TESTER :
liste_planetes = [mercure, venus, mars]

#Paramètres du graphique : --------------------------------------

#fig=plt.figure(figsize=(21/2.54, 29.7/2.54 ), tight_layout=True) # dimensions A4
fig=plt.figure(figsize=(12, 6), tight_layout=True, facecolor=couleurFond)

ax=plt.subplot()
plt.title(f"Distance des planètes à {nom_ref} (UA)", pad=10, fontsize=16, fontweight='bold')

moyenne_min = 35.0 # une valeur élevée, par défaut
for planete in liste_planetes:
    X = periode.utc_strftime()
    Y = distance(planete_reference, planete, periode)
    couleur, nom = attributs_planete(planete)
    moyenne = np.mean(Y)
    if moyenne < moyenne_min:
        proxima = nom
        moyenne_min = moyenne
        couleur_min = couleur
    ax.plot(X, Y, c=couleur, label=nom)
    ax.axhline(moyenne, color=couleur, ls='-.',lw=1)

ax.text(0.01,0.01,"la plus proche :", fontsize=12,
        transform=ax.transAxes)
ax.text(0.11,0.01,f"{proxima}", fontsize=12, fontweight='bold',
        color=couleur_min,
        transform=ax.transAxes)
ax.set_xlim(date_debut, date_fin)
ax.set_xticks([])
#----------------------------------------------------------------------------

#Affichage des dates :
texte=f"{NJours} jours\ndu " + date_JMA(periode[0]) +"\nau " +date_JMA(periode[-1])
plt.text(0.010, 0.98, texte, transform=ax.transAxes,
         c='white', va='top', fontsize=12, bbox=dict(fc='orangered'))

plt.legend(loc='upper right', fontsize=11)
ax.set_axisbelow(True)#place la grille sous les points tracés.

# signature
fig.text(1.012,0.,'David ALBERTO (www.astrolabe-science.fr)',rotation=90,
        ha='right',va='bottom',transform=ax.transAxes,
        fontsize=8)

fig.savefig(NomFichier + ".png", dpi=200)
fig.savefig(NomFichier + ".pdf")
# # plt.show()
