#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Créé le Mon Apr 18 19:47:48 2022

@auteur: david ALBERTO
(www.astrolabe-science.fr)

Documentation Skyfield : https://rhodesmill.org/skyfield/
"""
import matplotlib.pyplot as plt
from skyfield import api, almanac
import datetime as dt  # créaction de dates
import pytz  # module de fuseaux horaires
import matplotlib.patches as patches  # rectangles du tableau

# import locale#dates en français, si on veut modifier le code
# pour afficher le nom du mois dans la date.
# locale.setlocale(locale.LC_ALL, 'fr_FR.UTF-8')#dates en français

#Paramètres personnalisables :--------------------------
plt.rcParams["font.family"] = "Roboto"# ou autre police installée
plt.rcParams["font.size"] = 12
zone = pytz.timezone('Europe/Paris')# fuseau horaire Europe Centrale
annee_debut=2025
annee_fin=2035
# ------------------------------
periode=range(annee_debut,annee_fin+1)

ts = api.load.timescale()#crée une échelle de dates Skyfield

# (Télé)chargement du fichier d'ephémérides de440s.bsp (32 MB, période 1849-2150)
eph = api.load('../skyfield-ephemerides/de440s.bsp')# si le fichier est déjà chargé sur le disque (chemin à adapter).
# eph=api.load('de440s.bsp')# pour télécharger automatiquement le fichier, dans le dossier du fichier Python


evenements=['Équinoxe de printemps',
            "Solstice d'été",
            "Équinoxe d'automne",
            "Solstice d'hiver"]
mois=['mars','juin','septembre','décembre']

couleurs={#couleur selon le jour du mois
    19 : 'blue',
    20 : 'indigo' ,
    21 : 'royalblue',
    22 : 'darkmagenta',
    23 : 'steelblue'
    }
# -----------------------

#Paramètres du graphique    
fig=plt.figure(figsize=(21/2.54,29.7/2.54),
               tight_layout=True,# marges très réduites
               facecolor='lightblue',# couleur de fond
               )
ax=plt.subplot(111)
ax.set_title("Dates de début de saison (heure française) - Années %i - %i"%(annee_debut,annee_fin),
             fontsize=14)
ax.invert_yaxis()# axe y en sens inverse
ax.set_ylim(annee_fin+0.5,annee_debut-1.5)
ax.set_yticks(periode)
ax.set_xticks([])
ax.set_xlim([-0.5,3.5])# à adapter selon la taille de police

# -----------------------
for N, E in enumerate(evenements):  # en-têtes
    ax.text(N,annee_debut-1,evenements[N],
            fontsize=11,
            ha='center',
            fontweight='bold')
    ax.text(N,annee_debut-0.6,mois[N],
            fontsize=11,
            ha='center',)

#ligne des en-têtes :
ax.plot([-0.4,3.4],[annee_debut-0.40,annee_debut-0.40],c='darkorange')
# -----------------------

# Calcul et affichage des dates de début de saisons
for an in periode:
    t0=dt.datetime(an, 1, 1, tzinfo=zone)
    t1=dt.datetime(an, 12, 31, tzinfo=zone)
    
    #conversion des dates en temps Skyfield
    t0 = ts.from_datetime(t0)
    t1 = ts.from_datetime(t1)
    
    #recherche des dates de saisons durant la période
    t, y = almanac.find_discrete(t0, t1, almanac.seasons(eph))
    for yi, ti in zip(y, t):
        date=ti.astimezone(zone).strftime('à %Hh %Mmin %Ss')
        N_jour = float(ti.astimezone(zone).strftime('%d'))
        ax.text(yi,an,'le '+str(N_jour)[:2],c=couleurs[N_jour],
                ha='center',va='center',
                fontweight = 'bold',
                fontsize=13
                )
        ax.text(
            # yi+0.12,an,date,fontsize=11,
            yi, an + 0.25, date, fontsize=13,
                ha='center',
                va='center')
    if an%2==0:#une bande bleue en fond (1 année sur deux)
        ax.add_patch(patches.Rectangle(
        (-0.50, an+0.5),4,-1,
        facecolor = 'lightsteelblue', alpha=0.5,zorder=-1,
     )) 

fig.text(0.02,0.005,"D. Alberto (www.astrolabe-science.fr)",fontsize=8)

fig.savefig("DatesSaisons"+str(annee_debut)+'-'+str(annee_fin)+'.pdf',dpi=400)
fig.savefig("DatesSaisons"+str(annee_debut)+'-'+str(annee_fin)+'.png',dpi=400)
